/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface_2611.c
 * $Revision: 1.4.2.1 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright  2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/* Copyright 2003 - 2005 Intel Corporation. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *	Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *      Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *      Neither the name Intel Corporation nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "iegd.h"
#include "drmP.h"
#include "drm.h"

#include "iegd_drm.h"
#include "iegd_drv.h"
#include "psb_intregs.h"
#include "intelpci.h"

int drm_irq_install(drm_device_t *dev);

#if KERNEL2624

/* get intel_buffer_fops from the interface_###.c files */
extern struct file_operations intel_buffer_fops;

/* Global variable to keep track the amount of memory we are using */
static int memory = 0;

int intel_firstopen_2624(struct drm_device *dev)
{

	unsigned long resource_start;
	int rv;
	intel_device_private_t *priv;
	priv=(intel_device_private_t *)dev->dev_private;

	intel_postinit(&priv);
	psb_init(priv);
	dev->dev_private=priv;

	/*
	 * Map MMIO addresses so that the DRM can control interrupt support
	 */

	resource_start = pci_resource_start(dev->pdev, PSB_MMIO_RESOURCE);
	priv->vdc_reg = ioremap(resource_start + PSB_VDC_OFFSET, PSB_VDC_SIZE);
	if (!priv->vdc_reg) {
		/*
		 * Normally we'd want to unload the driver on failure.  But due
		 * to circular dependancies, we can only return failure.
		 */
		/* psb_driver_unload(dev); */
		return 1;
	}

	priv->sgx_reg = ioremap(resource_start + PSB_SGX_OFFSET, PSB_SGX_SIZE);
	if (!priv->sgx_reg) {
		/*
		 * Normally we'd want to unload the driver on failure.  But due
		 * to circular dependancies, we can only return failure.
		 */
		/* psb_driver_unload(dev); */
		return 1;
	}

	priv->msvdx_reg = ioremap(resource_start + PSB_MSVDX_OFFSET, PSB_MSVDX_SIZE);
	if (!priv->msvdx_reg) {
		/*
		 * Normally we'd want to unload the driver on failure.  But due
		 * to circular dependancies, we can only return failure.
		 */
		/* psb_driver_unload(dev); */
		return 1;
	}

	if (!dev->irq_enabled) {
		rv = drm_irq_install(dev);
		if (rv != 0) {
			DRM_ERROR("%s: could not install IRQs: rv = %d\n", __FUNCTION__, rv);
			return rv;
		}
	}

	return 0;

}

void intel_preclose_2624(drm_device_t *dev, struct drm_file *filp)
{
	intel_prerelease(dev);
}

/*
 * Implement the 2.6.24 kernel interface for the device specific IOCTL
 * that gets pages of memory from the DRM and returns them to the caller.
 */
int intel_getpages_2624(struct drm_device *dev,
		void *data,
		struct drm_file *filepriv)
{
	drm_intel_getpages_t *getpages;
	unsigned long bytes;
	int order;
	int size;

	unsigned long address;
	unsigned long phy_address;
	unsigned long offset;

	struct page *pg;

	unsigned long virtual;
	struct file_operations *old_fops;

	intel_device_private_t *dev_ptr = dev->dev_private;
	drm_intel_listpages_t *page;
	drm_intel_list_t     *list;

	DRM_DEBUG("\n");
	DRM_INFO("in intel_getpages_2624, calling intel_getpages\n");
	getpages = (drm_intel_getpages_t *)data;

	bytes = getpages->size;

	/*
	 * Check to see if this allocation would exceed 16MEG in total memory
	 * This is to prevent denial of service attack. 16Meg should be enough.
	 */
	if((memory + bytes) > MB(16) ){
		/* We exceeded 16MEG. Bail out */
		DRM_ERROR("Total memory allocated exceeded 16Meg!\n");
		DRM_INFO("Total memory allocated exceeded 16Meg!\n");
		return -EFAULT;
	}

	/* number of pages that are needed */
	size = bytes>>PAGE_SHIFT;
	if(bytes & ~(PAGE_SIZE*size)){
		++size;
	}
	order = ORDER(size);
	DRM_DEBUG("Allocating bytes:%#lx,size:%d,order:%d\n",
		(unsigned long)bytes,size,order); 

	/*
	 * Allocate the pages.
	 * returns kernel logical address. 
	 * Is this the same as the kernel virtual address?? 
	 */
	address = ALLOC_PAGES(order,0);
	if(!address){
		DRM_ERROR("Can't get pages\n");
		DRM_INFO("Can't get pages\n");
		return -EFAULT;
	}
	phy_address = __pa(address);

	/* Find virtual address of the phys address */
	pg = virt_to_page((void *)address);
	offset = pg->index;

	/* Find the number of bytes that is actually allocated */
	size = PAGE_SIZE<<order;
	DRM_DEBUG("Allocated address:%#lx,page offset:%#lx,phy_address:%#lx\n",
		address,offset,phy_address); 

	/*do_mmap on the logical address and return virtual address */
	down_write(&current->mm->mmap_sem);

	old_fops = filepriv->filp->f_op;
	filepriv->filp->f_op = &intel_buffer_fops;

	virtual = do_mmap(filepriv->filp, 0, size,
			PROT_READ|PROT_WRITE,MAP_SHARED, phy_address);
	filepriv->filp->f_op = old_fops;

	up_write(&current->mm->mmap_sem);
	DRM_DEBUG("Mmaped virtual:%#lx,address:%#lx\n",virtual,
		(unsigned long)__va(phy_address));

	if(virtual > -1024UL){
		DRM_ERROR("mmap failed:%d\n",(int)virtual);
		DRM_INFO("mmap failed:%d\n",(int)virtual);
		return -EFAULT;
	}

	getpages->phy_address = phy_address;
	getpages->virt_address = virtual;
	getpages->size = size;
	getpages->offset = offset;

	DRM_DEBUG("Mmap success requested size:%d (%d)\n",
		getpages->size,(int)bytes);

	/* alloc the page to be put into the linked list */
	page = ALLOC(sizeof(*page),DRM_MEM_DRIVER);
	if(!page){
		DRM_DEBUG("Can't alloc list for page\n");
		DRM_INFO("Can't alloc list for page\n");
		return -ENOMEM;
	}

	/*page->pid=current->pid;*/
	page->pid = current->group_leader->pid;
	page->size = size;
	page->phy_address = phy_address;
	page->virt_address = virtual;
	page->offset = offset;

	DRM_DEBUG("parent pid:%d,pid:%d,group_leader->pid:%d\n"
		,current->parent->pid,current->pid,current->group_leader->pid);

	/* Alloc the list to be added then add it to the linked list */
	list = ALLOC(sizeof(*list),DRM_MEM_DRIVER);
	if(!list){
		DRM_DEBUG("Can't alloc list for page\n");
		DRM_INFO("Can't alloc list for page\n");
		FREE(page,sizeof(*page),0);
		return -ENOMEM;
	}
	memset(list,0,sizeof(*list));
	list->page = page;
	LOCK_DRM(dev);
	list_add(&list->head,&dev_ptr->pagelist->head);
	UNLOCK_DRM(dev);

	/* update the total amount of memory we use */
	memory += size;
	DRM_DEBUG("memory has:%d bytes\n",memory);

	DRM_INFO("intel_getpages Exit\n");
	return 0;
}


/*
 * Implement the 2.6.24 kernel interface for the device specific IOCTL
 * that frees pages of memory that were previouslly allocated from the DRM.
 */
int intel_freepages_2624(struct drm_device *dev,
		void *data,
		struct drm_file *filepriv)
{
	drm_intel_freepages_t *freepages;
	unsigned long bytes;
	int order;
	int size;

	intel_device_private_t *dev_ptr=dev->dev_private;
	drm_intel_listpages_t *page;
	drm_intel_list_t *r_list=NULL;
	struct list_head *pagelist;

	DRM_DEBUG("Freeing pages\n");
	freepages = (drm_intel_freepages_t *)data;

	/* number of pages that are needed */
	bytes = freepages->size;
	size = bytes>>PAGE_SHIFT;
	if(bytes & ~(PAGE_SIZE*size)){
		++size;
	}
	order = ORDER(size);
	DRM_DEBUG("bytes:%d,size:%d,order:%d,phy_address:%#lx\n", (int)bytes,
			(int)size,(int)order,freepages->phy_address);

	/* free the pages */
	DRM_DEBUG("freeing address:%#lx,size:%#lx\n",
			(unsigned long)__va(freepages->phy_address),(unsigned long)bytes);

	DRM_DEBUG("parent pid:%d,pid:%d,group_leader->pid:%d\n"
		,current->parent->pid,current->pid,current->group_leader->pid);

	/* See if the requested address is in our page list */
	LOCK_DRM(dev);
	pagelist = &dev_ptr->pagelist->head;
	list_for_each(pagelist, &dev_ptr->pagelist->head){
		r_list=list_entry(pagelist, drm_intel_list_t, head);
		if((r_list->page->pid==current->group_leader->pid)
			&& (r_list->page->phy_address==freepages->phy_address)){

		DRM_DEBUG("found pid:%d\n",current->group_leader->pid);
		DRM_DEBUG("size:%d\n",r_list->page->size);
		DRM_DEBUG("phy_address:%#lx\n",r_list->page->phy_address);
		DRM_DEBUG("virt_add:%#lx\n",r_list->page->virt_address);
		DRM_DEBUG("offset:%#lx\n",r_list->page->offset);

		break;
		}

	}

	if(pagelist == (&dev_ptr->pagelist->head)){
		DRM_DEBUG("Can't find pages alloc for pid:%d\n",current->pid);
		UNLOCK_DRM(dev);
		return -EINVAL;
	}

	/* munmap the region 1st */
	down_write(&current->mm->mmap_sem);
	DRM_DEBUG("Unmapping virt_address:%#lx\n",freepages->virt_address);
	do_munmap(current->mm,freepages->virt_address,bytes);
	up_write(&current->mm->mmap_sem);

	/* Free the pages! */
	FREE_PAGES((unsigned long)__va(freepages->phy_address), order, 0);

	/* Free the page list */
	page = r_list->page;
	list_del(pagelist);
	size = r_list->page->size;
	FREE(pagelist,sizeof(*pagelist),0);
	FREE(page,sizeof(*page),0);
	UNLOCK_DRM(dev);

	/* update the total memory that we use */
	memory -= size;
	DRM_DEBUG("memory has:%d bytes\n", memory);
	return 0;
}


/*
 * Implement the 2.6.24 kernel interface for the device specific IOCTL
 * that stores client specific information.
 */
int intel_drm_info_init_2624(struct drm_device *dev,
		void *data,
		struct drm_file *filepriv)
{
	intel_drm_info_t *info;
	intel_drm_info_t *info_ptr;
	intel_device_private_t *dev_ptr;

	if (dev == NULL) {
		DRM_INFO("ERROR ERROR, drm device is NULL\n");
		return -EFAULT;
	}
	DRM_DEBUG("info init succesful dev_private:%#lx\n",
			(unsigned long)dev->dev_private);
	dev_ptr = dev->dev_private;

	/* See if dev_private is already allocated */
	if(!dev->dev_private){
		DRM_ERROR("dev_private not allocated!\n");
		return 0;
	}
	info_ptr = dev_ptr->info_ptr;

	/* See if info is already allocated */
	if(info_ptr->device_id){
		DRM_DEBUG("Info already allocated: device id = 0x%lx\n",
				info_ptr->device_id);
		DRM_ERROR("Info already allocated!\n");
		return 0;
	}

	info = (intel_drm_info_t *)data;

	info_ptr->device_id = info->device_id;
	info_ptr->revision = info->revision;
	info_ptr->video_memory_offset = info->video_memory_offset;
	info_ptr->video_memory_size = info->video_memory_size;
	info_ptr->hw_status_offset = info->hw_status_offset;
	DRM_DEBUG("Saving dev_id:%#lx  rev:%#lx  offset:%#lx  size:%#lx,  "
			"hwst_offset:%lx\n",
		info_ptr->device_id, info_ptr->revision,
		info_ptr->video_memory_offset, info_ptr->video_memory_size,
		info_ptr->hw_status_offset);

	return 0;
}


/*
 * Implement the 2.6.24 kernel interface for the device specific IOCTL
 * that retrieves client specific information.
 */
int intel_drm_info_get_2624(struct drm_device *dev,
		void *data,
		struct drm_file *filepriv)
{
	intel_drm_info_t *info;
	intel_device_private_t *dev_ptr = dev->dev_private;
	intel_drm_info_t *info_ptr = dev_ptr->info_ptr;

	DRM_DEBUG("Info get dev_id:%#lx  rev:%#lx  offset:%#lx  size:%#lx  "
			"hwst_offset:%lx\n",
		info_ptr->device_id,info_ptr->revision,
		info_ptr->video_memory_offset,info_ptr->video_memory_size,
		info_ptr->hw_status_offset);

	info = (intel_drm_info_t *)data;

	info->device_id = info_ptr->device_id;
	info->revision = info_ptr->revision;
	info->video_memory_offset = info_ptr->video_memory_offset;
	info->video_memory_size = info_ptr->video_memory_size;
	info->hw_status_offset = info_ptr->hw_status_offset;

	return 0;
}

/*
 * The following 2 functions were taken from drm_memory.c 
 * For some reason they are not being exported to use by the other drm.
 */

/**
 * Allocate pages.
 *
 * \param order size order.
 * \param area memory area. (Not used.)
 * \return page address on success, or zero on failure.
 *
 * Allocate and reserve free pages.
 */
unsigned long intel_alloc_pages(int order, int area)
{
	unsigned long address;
	unsigned long bytes	  = PAGE_SIZE << order;
	unsigned long addr;
	unsigned int  sz;

	address = __get_free_pages(GFP_KERNEL, order);
	if (!address) 
		return 0;

				/* Zero */
	memset((void *)address, 0, bytes);

				/* Reserve */
	for (addr = address, sz = bytes;
	     sz > 0;
	     addr += PAGE_SIZE, sz -= PAGE_SIZE) {
		SetPageReserved(virt_to_page(addr));
	}

	return address;
}

/**
 * Free pages.
 *
 * \param address address of the pages to free.
 * \param order size order.
 * \param area memory area. (Not used.)
 *
 * Unreserve and free pages allocated by alloc_pages().
 */
void intel_free_pages(unsigned long address, int order, int area)
{
	unsigned long bytes = PAGE_SIZE << order;
	unsigned long addr;
	unsigned int  sz;

	if (!address) {
		return;
	}

	/* Unreserve */
	for (addr = address, sz = bytes; sz > 0;
			addr += PAGE_SIZE, sz -= PAGE_SIZE) {
		ClearPageReserved(virt_to_page(addr));
	}

	free_pages(address, order);
}

static int psb_driver_load(struct drm_device *dev, unsigned long chipset)
{
	intel_device_private_t *priv;

	priv=(intel_device_private_t *)dev->dev_private;

	return 0;
}

int intel_drm_plb_interrupts_2624 (struct drm_device *dev,
		void *data,
		struct drm_file *filepriv)
{
	intel_device_private_t *priv = dev->dev_private;
	interrupt_info_t *plb_info;
	unsigned long irqflags;
	int ret = 0;

	plb_info = (interrupt_info_t *)data;

	/* USW15 definition of in and out
	 *
	 * in/out[0] VDC
	 * in/out[1] sgx
	 * in/out[2] sgx2
	 * in/out[3] msvdx
	 */

	plb_info->out[0]=0;
	plb_info->out[1]=0;
	plb_info->out[2]=0;
	plb_info->out[3]=0;

	switch (plb_info->req_type) {
		case CLEAR_INT:
			plb_info->in[0] &= priv->vdc_irq_mask;
			plb_info->in[1] &= priv->sgx_irq_mask;
			plb_info->in[2] &= priv->sgx_irq_mask2;
			plb_info->in[3] &= priv->msvdx_irq_mask;

			if (plb_info->in[0] || plb_info->in[1] ||
					plb_info->in[2] || plb_info->in[3]) {

				spin_lock_irqsave(&priv->irqmask_lock, irqflags);
				priv->out_vdc &= ~plb_info->in[0];
				plb_info->out[0] = priv->out_vdc;

				priv->out_sgx &= ~plb_info->in[1];
				plb_info->out[1] = priv->out_sgx;

				priv->out_sgx2 &= ~plb_info->in[2];
				plb_info->out[2] = priv->out_sgx2;

				priv->out_mtx &= ~plb_info->in[3];
				plb_info->out[3] = priv->out_mtx;
				spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

				plb_info->req_status = INT_CLEARED;

			} else {
				plb_info->req_status = INT_NOOP;
			}

			break;

		case READ_INT:
			plb_info->out[0] = priv->out_vdc;
			plb_info->out[1] = priv->out_sgx;
			plb_info->out[2] = priv->out_sgx2;
			plb_info->out[3] = priv->out_mtx;
			plb_info->req_status = INT_READ;

			break;

		case WAIT_INT:
			plb_info->in[0] &= priv->vdc_irq_mask;
			plb_info->in[1] &= priv->sgx_irq_mask;
			plb_info->in[2] &= priv->sgx_irq_mask2;
			plb_info->in[3] &= priv->msvdx_irq_mask;

			if (plb_info->in[0] || plb_info->in[1] ||
					plb_info->in[2] || plb_info->in[3]) {

				spin_lock_irqsave(&priv->irqmask_lock, irqflags);

				/* none of the interrupts have ocurred */
				if ((priv->out_vdc & plb_info->in[0]) ||
						(priv->out_sgx & plb_info->in[1]) ||
						(priv->out_sgx2 & plb_info->in[2]) ||
						(priv->out_mtx & plb_info->in[3])) {

					/* At least one of the interrupts has already occurred */
					plb_info->req_status = INT_STORED;

				} else {

					/* Wait for an interrupt to occur */
					priv->event_present = 0;
					spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

					DRM_WAIT_ON(ret, priv->event_queue, 20 * DRM_HZ,
							priv->event_present);

					if (ret) {
						plb_info->req_status = INT_TIMEOUT;
						break;
					}

					spin_lock_irqsave(&priv->irqmask_lock, irqflags);

					plb_info->req_status = INT_HANDLED;

				}
				plb_info->out[0] = priv->out_vdc;
				plb_info->out[1] = priv->out_sgx;
				plb_info->out[2] = priv->out_sgx2;
				plb_info->out[3] = priv->out_mtx;

				/* Clear the outstanding interrupts that have just been
				 * retrieved
				 */
				priv->out_vdc &= ~(plb_info->out[0] & plb_info->in[0]);
				priv->out_sgx &= ~(plb_info->out[1] & plb_info->in[1]) ;
				priv->out_sgx2 &= ~(plb_info->out[2] & plb_info->in[2]);
				priv->out_mtx &= ~(plb_info->out[3] & plb_info->in[3]);
				spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

			} else {

				/* Unsupported interrupt */
				plb_info->req_status = INT_NOOP;

			}

			break;

		case UNMASK_INT:
			spin_lock_irqsave(&priv->irqmask_lock, irqflags);
			PSB_WVDC32(0x00000000, IMR);
			spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

			break;

		case MASK_INT:

			spin_lock_irqsave(&priv->irqmask_lock, irqflags);
			PSB_WVDC32(0xFFFFFFFF, IMR);
			spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

			break;

		default:

			plb_info->req_status = INT_INVALID;
	}

	return 0;
}


drm_ioctl_desc_t intel_ioctls[] = {
	DRM_IOCTL_DEF(DRM_INTEL_GETPAGES, intel_getpages_2624, 0),
	DRM_IOCTL_DEF(DRM_INTEL_FREEPAGES, intel_freepages_2624, 0),
	DRM_IOCTL_DEF(DRM_INTEL_INFO_INIT, intel_drm_info_init_2624, 0),
	DRM_IOCTL_DEF(DRM_INTEL_INFO_GET, intel_drm_info_get_2624, 0),
	DRM_IOCTL_DEF(DRM_INTEL_INTERRUPT, intel_drm_plb_interrupts_2624, 0)
};

int intel_max_ioctl = DRM_ARRAY_SIZE(intel_ioctls);



static struct pci_device_id pciidlist[] = {
	INTEL_PCI_IDS
};

int device_is_agp_2624(drm_device_t * dev)
{
	return 1;
}

static struct drm_driver driver = {
	.firstopen = intel_firstopen_2624,
	.preclose = intel_preclose_2624,
	.reclaim_buffers=drm_core_reclaim_buffers,
	.get_map_ofs=drm_core_get_map_ofs,
	.get_reg_ofs=drm_core_get_reg_ofs,

	.device_is_agp = device_is_agp_2624,

	.major = DRIVER_MAJOR,
	.minor = DRIVER_MINOR,
	.patchlevel = DRIVER_PATCHLEVEL,
	.name = DRIVER_NAME,
	.desc = DRIVER_DESC,
	.date = DRIVER_DATE,

	.driver_features = DRIVER_USE_AGP|DRIVER_REQUIRE_AGP|DRIVER_USE_MTRR,
	.ioctls = intel_ioctls,
	.fops = {
		.owner = THIS_MODULE,
		.open = drm_open,
		.release = drm_release,
		.ioctl = drm_ioctl,
		.mmap = drm_mmap,
		.poll = drm_poll,
		.fasync = drm_fasync,
	},
	.pci_driver = {
		.name          = DRIVER_NAME,
		.id_table      = pciidlist,
	}
};

static struct drm_driver driver_plb = {
	.load = psb_driver_load,
	.firstopen = intel_firstopen_2624,
	.preclose = intel_preclose_2624,
	.reclaim_buffers=drm_core_reclaim_buffers,
	.get_map_ofs=drm_core_get_map_ofs,
	.get_reg_ofs=drm_core_get_reg_ofs,

	.device_is_agp = device_is_agp_2624,

	.major = DRIVER_MAJOR,
	.minor = DRIVER_MINOR,
	.patchlevel = DRIVER_PATCHLEVEL,
	.name = DRIVER_NAME,
	.desc = DRIVER_DESC,
	.date = DRIVER_DATE,

	.driver_features = DRIVER_USE_AGP | DRIVER_REQUIRE_AGP |
		DRIVER_HAVE_IRQ | DRIVER_IRQ_SHARED | DRIVER_USE_MTRR,
	.ioctls = intel_ioctls,
	.irq_preinstall = psb_irq_preinstall,
	.irq_postinstall = psb_irq_postinstall,
	.irq_uninstall = psb_irq_uninstall,
	.irq_handler = psb_irq_handler,

	.fops = {
		.owner = THIS_MODULE,
		.open = drm_open,
		.release = drm_release,
		.ioctl = drm_ioctl,
		.mmap = drm_plb_mmap,
		.poll = drm_poll,
		.fasync = drm_fasync,
	},
	.pci_driver = {
		.name          = DRIVER_NAME,
		.id_table      = pciidlist,
	}
};

int intel_init(void)
{
	driver.num_ioctls = intel_max_ioctl;
	driver_plb.num_ioctls = intel_max_ioctl;

	/* We are peeking into the global AGP structures that
	 * we have access to in order to determine what chipset we're
	 * on.  This isn't necessarily a good thing to do.
	 */

	if (gart_id->device_id == PCI_DEVICE_ID_PLB) {
		printk(KERN_ERR "Initializing DRM for Intel US15 SCH\n");
		return drm_init(&driver_plb);
	} else {
		return drm_init(&driver);
	}

}

void intel_exit(void)
{
	drm_exit(&driver);
}

struct file_operations intel_buffer_fops = {
	.open	 = drm_open,
	.release = drm_release,
	.ioctl	 = drm_ioctl,
	.mmap	 = intel_mmap_buffers,
	.poll = drm_poll,
	.fasync  = drm_fasync,
};
#endif
