/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface.c
 * $Revision: 1.30 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright © 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "global.h"
#include "intelpci.h"
#include <linux/pagemap.h>
#include <linux/list.h>

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,5)

static int iegd_plb_fetch_size(void);
static void iegd_plb_tlbflush(struct agp_memory *mem);
static int iegd_plb_init_gtt(u32 order);
static int AGP_CREATE_GATT(iegd_plb_create_gatt_table);
static void iegd_plb_cleanup(void);
static struct page *iegd_plb_vm_nopage(struct vm_area_struct *,
	unsigned long, int *);
static void iegd_plb_vm_close(struct vm_area_struct *);
int iegd_plb_insert_entries(struct agp_memory *, off_t, int);
int iegd_plb_remove_entries(struct agp_memory *, off_t, int);
void iegd_plb_free_by_type(struct agp_memory *);
int iegd_plb_configure(void);


/* Each structure in this array contains three elements:
 * Size of GTT in KB
 * Number of 32-bit entries that make up the GTT
 * Page "order" -- 2^order == number of contiguous CPU pages
 * required to store the GTT
 */
struct aper_size_info_fixed iegd_plb_sizes[] =
{
	{256, 65536, 6},
};

bridge_driver_t drv_plb = {
	.owner              = THIS_MODULE,
	.size_type          = FIXED_APER_SIZE,
	.aperture_sizes     = iegd_plb_sizes,
	.num_aperture_sizes = 1,
	.needs_scratch_page = TRUE,
	.configure          = iegd_plb_configure,
	.fetch_size         = iegd_plb_fetch_size,
	.cleanup            = iegd_plb_cleanup,
	.tlb_flush          = iegd_plb_tlbflush,
	.mask_memory        = iegd_cmn_mask_memory,
	.masks              = iegd_cmn_masks,
	.agp_enable         = iegd_cmn_agp_enable,
	.cache_flush        = global_cache_flush,
	.create_gatt_table  = iegd_plb_create_gatt_table,
	.free_gatt_table    = iegd_cmn_free_gatt_table,
	.insert_memory      = iegd_plb_insert_entries,
	.remove_memory      = iegd_plb_remove_entries,
	.alloc_by_type      = iegd_cmn_alloc_by_type,
	.free_by_type       = iegd_plb_free_by_type,
	.agp_alloc_page     = agp_generic_alloc_page,
	.agp_destroy_page   = agp_generic_destroy_page,
};

struct vm_operations_struct iegd_plb_vm_ops = {
	.nopage = iegd_plb_vm_nopage,
	.close = iegd_plb_vm_close
};

static DECLARE_MUTEX(client_sem);

struct client_list_struct {
	struct list_head list;
	struct vm_area_struct *vma;
	pid_t pid;
};

static LIST_HEAD(client_list);


static int iegd_plb_fetch_size()
{
	struct aper_size_info_fixed *values;

	values = A_SIZE_FIX(agp_bridge->driver->aperture_sizes);


	agp_bridge->previous_size = agp_bridge->current_size =
		(void *)(values);

	return values[0].size;
}

static void iegd_plb_tlbflush(struct agp_memory *mem)
{
	u32 sgx_mmu;

	/* Flush TLB */
	sgx_mmu = readl(private_data.registers + 0x40C00);
	sgx_mmu &= 0xFFFFFFE0;
	sgx_mmu |= 0x0C;
	writel(sgx_mmu, private_data.registers + 0x40C00);

	wmb();
	sgx_mmu = readl(private_data.registers + 0x40C00);
	sgx_mmu &= 0xFFFFFFE0;
	writel(sgx_mmu, private_data.registers + 0x40C00);

	return;
}

#define IUS15_GMCH_MSAC 0x62

static int iegd_plb_init_gtt(u32 order)
{
	u32 gtt_pgctl_reg;
	u32 gtt_bus_addr;
	u32 gtt_enabled = FALSE;
	int num_entries;
	u32 *gtt_table, *dstvirt;
	u32 *sgx_dir, sgx_mmu;
	u32 iegd_scratch, aperphys;
	u8 temp;
	struct page *gtt_table_page;
	int i,j;

	/* Has the system BIOS only allocateda GTT for 128MB?  If
	 * so we need to replace it with one sized for 256MB
	 */
	pci_read_config_byte(private_data.pdev, IUS15_GMCH_MSAC, &temp);
	if ((temp & 0x03) == 0x03) {
		AGN_DEBUG("Graphics aperture is configured for 128MB");
		AGN_DEBUG("Enabling 256MB split aperture");
		private_data.split_gtt = 1;
	} else {
		private_data.split_gtt = 0;
	}

	gtt_pgctl_reg = readl(private_data.registers +
		I810_PGETBL_CTL);
	global_cache_flush();
	gtt_bus_addr = gtt_pgctl_reg & 0xFFFFF000;
	gtt_enabled  = gtt_pgctl_reg & I810_PGETBL_ENABLED;

	/* we have to call this as early as possible after the MMIO base
	 * address is known */
	iegd_cmn_init_gtt_entries();

	/* Update the scratch registers to say that we have no stolen memory */
	iegd_scratch = readl(private_data.registers + 0x71410);
	if ((iegd_scratch & 0xE1DF0000) == 0xE1DF0000) {
		/* if our vBios modify only the stolen memory bit */
		iegd_scratch |= 0x00000004;
		writel(iegd_scratch, private_data.registers + 0x71410);
	} else {
		/* Not our vBIOS but set the stolen memory anyway */
		writel(0xE1DF0004, private_data.registers + 0x71410);
	}

	/* Reportthat we have 0 stolen memory regardless of what was
	 * really in there.  We _want_ to insert fresh pages on top of
	 * stolen memory. */
	writel(0, private_data.registers + 0x71418);

	num_entries = (1 << order) * KB(1);

	private_data.upper_gtt=NULL;

	/*
	 * If GTT not enabled created our own gtt table from kernel memory
	 * and initialize it to scratch page. This in case the VBIOS is
	 * not our VBIOS
	 */
	if (!gtt_enabled) {
		gtt_table = (u32 *)__get_free_pages(GFP_KERNEL, order);

		/* Make sure allocation was successful */
		if (NULL == gtt_table) {
			AGN_ERROR("Failed to allocate kernel pages");
			return (-ENOMEM);
		}

		for (i=0; i < (1 << order); i++) {
				dstvirt = gtt_table + (PAGE_SIZE * i);
				gtt_table_page = virt_to_page(dstvirt);
				AGN_DEBUG("Setting reserved bit on %p", gtt_table_page);
				set_bit(PG_reserved, &gtt_table_page->flags);
		}

		private_data.upper_gtt = gtt_table + 0x8000;
		agp_bridge->gatt_bus_addr = virt_to_phys(gtt_table);

		for (i = 0; i < num_entries; i++) {
			gtt_table[i] = (unsigned long) agp_bridge->scratch_page;
		}

		/* Enable the newly created GTT */
		AGN_DEBUG("Enabling new GTT");
		writel(agp_bridge->gatt_bus_addr | I810_PGETBL_ENABLED,
			private_data.registers+I810_PGETBL_CTL);
		readl(private_data.registers+I810_PGETBL_CTL);

	} else if (private_data.split_gtt) {

		/* We're keeping the system BIOS created normal gtt but
		 * augmenting it with more entries
		 */
		gtt_table = (u32 *)__get_free_pages(GFP_KERNEL, order - 1);

		AGN_DEBUG("Allocated secondary GTT at %p:%p (virt:phys)", gtt_table,
			virt_to_phys(gtt_table));

		/* Make sure allocation was successful */
		if (NULL == gtt_table) {
			AGN_ERROR("Failed to allocate kernel pages");
			return (-ENOMEM);
		}

		private_data.upper_gtt = gtt_table;

		for (i = 0; i < num_entries; i++) {
			gtt_table[i] = (unsigned long) agp_bridge->scratch_page;
		}

		agp_bridge->gatt_bus_addr = gtt_bus_addr;

	} else {

		agp_bridge->gatt_bus_addr = gtt_bus_addr;

	}

	/*
	 * Now that the GTT exists and has been configured, enable
	 * the SGX MMU to point to the GTT as its page tables
	 */

	/* The directory level is a single page of memory */
	sgx_dir = (u32 *)__get_free_pages(GFP_KERNEL, 0);
	if (NULL == sgx_dir ) {
		AGN_ERROR("Failed to allocate kernel page");
		return (-ENOMEM);
	}

	/* Mark the directory so that it is not swappable */
	gtt_table_page = virt_to_page( sgx_dir );
	set_bit(PG_reserved, &gtt_table_page->flags);

	memset (sgx_dir, 0, PAGE_SIZE);

	/* Initialize the directory so that each used page table
	 * is addressed
	 */

	/* Make sure entire SGX directory is populated */
	for (i = 0; i < 0x400; i++) {
		sgx_dir[i] = agp_bridge->gatt_bus_addr | 0x01;
	}

	pci_read_config_dword(private_data.pdev, I915_GMADDR, &aperphys);
	aperphys &= PCI_BASE_ADDRESS_MEM_MASK;
	aperphys = aperphys >> 22;

	for (i = 0; i < (1 << order); i++) {
		/* Set the address for 2D/3D*/
		sgx_dir[i] = agp_bridge->gatt_bus_addr + (PAGE_SIZE * i);
		/* Set the address for hostport */
		sgx_dir[i+aperphys] = agp_bridge->gatt_bus_addr + (PAGE_SIZE * i);

		/* Mark them as valid */
		sgx_dir[i] |= 0x01;
		sgx_dir[i+aperphys] |= 0x01;

		AGN_DEBUG("Directory %d is %08lx", i, sgx_dir[i]);
	}

	/* If we're in split gtt mode, set the directory entries of the second
	 * gtt
	 */

	if (private_data.split_gtt) {
		j=0;
		for (i = (1 << (order - 1)); i < (1 << order); i++) {
			/* Set the address for 2D/3D*/
			sgx_dir[i] = virt_to_phys(private_data.upper_gtt) + (PAGE_SIZE * j);
			/* Set the address for hostport */
			sgx_dir[i+aperphys] = virt_to_phys(private_data.upper_gtt) + (PAGE_SIZE * j);

			j++;

			/* Mark them as valid */
			sgx_dir[i] |= 0x01;
			sgx_dir[i+aperphys] |= 0x01;
			AGN_DEBUG("Directory %d is %08lx", i, sgx_dir[i]);
		}
	}

	/*
	 * Program the directory's address into the MMU control
	 * register
	 */

	/* Flush the cache */
	flush_cache_all();
	global_cache_flush();

	/* Invalidate directory cache */
	sgx_mmu = readl(private_data.registers + 0x40C00);
	sgx_mmu |= 0x1E;
	writel(sgx_mmu, private_data.registers + 0x40C00);
	wmb();
	readl(private_data.registers + 0x40C00);

	writel(virt_to_phys(sgx_dir), private_data.registers + 0x40C84);
	wmb();
	readl(private_data.registers + 0x40C84);

	/* Turn on host access to aperture via the MMU */
	sgx_mmu = readl(private_data.registers + 0x40C00);
	sgx_mmu &= 0xFFFE0000;
	writel(sgx_mmu, private_data.registers + 0x40C00);
	wmb();
	readl(private_data.registers + 0x40C00);

	return 0;
}


static int AGP_CREATE_GATT(iegd_plb_create_gatt_table)
{
	u32 order;
	u32 mmio_bus_addr, temp2;
	int ret;

	agp_bridge->gatt_table_real = NULL;

	order=A_SIZE_FIX(agp_bridge->current_size)->page_order;

	/* Find and save the address of the MMIO register */
	pci_read_config_dword(private_data.pdev, I915_MMADDR,
		&mmio_bus_addr);
	mmio_bus_addr &= 0xFFF80000;

	private_data.registers = (volatile u8 *) ioremap(mmio_bus_addr,
		KB(512));

	if (!private_data.registers) {
		AGN_ERROR("ioremap failed to map mmio");
		return (-ENOMEM);
	}

	pci_read_config_dword(private_data.pdev, I915_PTEADDR, &temp2);

	/* FIXME: double check the size of area to map to pci space */
	private_data.gtt = (volatile u32 *)ioremap(temp2,
			A_SIZE_FIX(agp_bridge->current_size)->num_entries * sizeof(u32));

	if (!private_data.gtt) {
		AGN_ERROR("ioremap failed to map gtt");
		return (-ENOMEM);
	}

	if((ret = iegd_plb_init_gtt(order))) {
		return (ret);
	}

	agp_bridge->gatt_table = NULL;

	return (0);
}

static void iegd_plb_cleanup(void)
{

	iounmap((void *)private_data.gtt);
	iounmap((void *)private_data.registers);
}


static void iegd_plb_vm_close(struct vm_area_struct *vma)
{
	struct list_head *tmp;
	struct client_list_struct *entry;

	down(&client_sem);
	list_for_each(tmp, &client_list) {
		entry = list_entry(tmp, struct client_list_struct, list);
		if (entry->vma == vma) {
			list_del(&entry->list);
			kfree(entry);
			AGN_DEBUG("Removed VMA %p from client list", vma);
			break;
		}
	}
	up(&client_sem);
}


static struct page *iegd_plb_vm_nopage(struct vm_area_struct *vma,
	unsigned long address,
	int *type)
{
	unsigned long offset=0;
	unsigned long physaddr=0;
	struct page *page;
	struct list_head *tmp;
	struct client_list_struct *entry;
	int flag=0;

	/* On the Intel SCH US15, we don't have a traditional aperture.  As
	 * a result, we're substituting the base of stolen memory
	 * as the aperture address.
	 *
	 * Mmaps relative to the base of stolen memory will be
	 * treated as mmaps covering parts of our virtual aperture.
	 *
	 * Given that a single surface may be mapped, and not the
	 * whole virtual aperture, we must translate the values
	 * received so that they are relative to our 0-based virtual
	 * aperture.
	 */
	offset = (vma->vm_pgoff << PAGE_SHIFT) - agp_bridge->gart_bus_addr;

	/* All pages returned must be noncached or write-combined*/
	if (agp_use_pat()) {
		pgprot_val(vma->vm_page_prot) &= ~(_PAGE_PCD | _PAGE_PWT);
		pgprot_val(vma->vm_page_prot) |= _PAGE_PAT;
	} else {
		vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	}

	/* Access to the virtual frame buffer does not appear to
	 * call open properly before faulting.  As a result, we
	 * need to do this housekeeping at each fault.
	 */
	down(&client_sem);
	list_for_each(tmp, &client_list) {
		entry = list_entry(tmp, struct client_list_struct, list);
		if (entry->vma == vma) {
			flag=1;
		}
	}

	if (!flag) {
		entry = kmalloc(sizeof(struct client_list_struct), GFP_KERNEL);
		if (entry) {
			entry->vma = vma;
			list_add(&(entry->list), &client_list);
			AGN_DEBUG("Added VMA %p to client list", vma);

			AGN_DEBUG("Scratch:  %p", virt_to_page(agp_bridge->scratch_page));

		} else {
			AGN_ERROR("Failed to add VMA to client list");
		}
	}
	up(&client_sem);

	offset += address - vma->vm_start;

	if (private_data.split_gtt && ((offset >> PAGE_SHIFT)) >= 0x8000) {
		physaddr = readl(private_data.upper_gtt + (offset >> PAGE_SHIFT)
			- 0x8000);
	} else {
	physaddr = readl(private_data.gtt + (offset >> PAGE_SHIFT));
	}


	physaddr &= PAGE_MASK;

	if (!pfn_valid(physaddr >> PAGE_SHIFT)) {
		AGN_ERROR("Referencing non-existant struct page.\n");
	}

	if (physaddr >= agp_bridge->gart_bus_addr) {
		AGN_DEBUG("Faulted before insert, returning scratch page");
		page = virt_to_page(__va(agp_bridge->scratch_page));
	} else {
		page = virt_to_page(__va(physaddr));
	}

	get_page(page);

	if (type) {
		*type = VM_FAULT_MINOR;
	}

	return (page);
}


int iegd_plb_insert_entries(struct agp_memory *mem,
	off_t pg_start, int type)
{
	int i,j,num_entries, zap;
	void *temp;
	struct list_head *tmp;
	struct client_list_struct *entry;
	unsigned long addr_start=0;
	unsigned long addr_end=0;
	unsigned long addr_offset=0;
	unsigned long vaddr;
	char *srcvirt;
	unsigned long srcphys;
	unsigned long dstphys;
	pgd_t *pgd;
	pud_t *pud;
	pmd_t *pmd;
	pte_t *pte;

	temp = agp_bridge->current_size;
	num_entries = A_SIZE_FIX(temp)->num_entries;

	/* If we try to write beyond gtt table, return error */
	if ((pg_start + mem->page_count) > num_entries) {
		AGN_ERROR("Trying to write beyond aperture limit");
		AGN_DEBUG("pg_start=0x%.8lx, mem->page_count=%d,"
				"num_entries=%d", pg_start, mem->page_count,
				num_entries);
		return -EINVAL;
	}

	/* The i830 can't check the GTT for entries since its read only,
	 * depend on the caller to make the correct offset decisions.
	 */

	if ((type != 0 && type != AGP_PHYS_MEMORY) ||
		(mem->type != 0 && mem->type != AGP_PHYS_MEMORY)) {
		AGN_ERROR("Unsupported memory type");
		AGN_DEBUG("mem->type=%x, type=%x", mem->type, type);
		return -EINVAL;
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	for (i = 0, j = pg_start; i < mem->page_count; i++, j++) {

		/* If we're inserting into stolen memory, we need to read
		 * the contents of the original page that occupied this space
		 */
		if (j < private_data.gtt_entries) {
			srcphys=readl(private_data.gtt+j);
			srcphys &= PAGE_MASK;

			if (srcphys >= agp_bridge->gart_bus_addr) {
				srcvirt=ioremap(srcphys, PAGE_SIZE);

				if (!srcvirt) {
					AGN_ERROR("Could not map stolen memory source %d:%08lX", j, srcphys);
					return -ENOMEM;
				}

				dstphys=AGP_MASK_GTT();
				dstphys &= PAGE_MASK;

				copy_page(__va(dstphys), srcvirt);

				iounmap(srcvirt);
			} else {
				AGN_ERROR ("Tried to copy a page not in stolen memory %d:%08lX", j, srcphys);
			}
		}

		if (private_data.split_gtt && (j >= 0x8000)) {
			writel(AGP_MASK_GTT(), private_data.upper_gtt + j - 0x8000);
		} else {
		writel(AGP_MASK_GTT(), private_data.gtt+j);
		readl(private_data.gtt+j);	/* PCI Posting. */
		}

		down(&client_sem);
		list_for_each(tmp, &client_list) {
			entry = list_entry(tmp, struct client_list_struct, list);

			/* We need to handle invalidating VMA's that are only mapping
			 * a portion of the virtual aperture.  Calculate what if
			 * any invalidated pages need to be zapped
			 */
			addr_start = (entry->vma->vm_pgoff << PAGE_SHIFT)
				- agp_bridge->gart_bus_addr;
			addr_end = addr_start + (entry->vma->vm_end - entry->vma->vm_start);
			addr_offset = j << PAGE_SHIFT;

			vaddr = entry->vma->vm_start + (addr_offset - addr_start);

			zap=0;
			pgd=NULL;
			pud=NULL;
			pmd=NULL;
			pte=NULL;

			pgd = pgd_offset(entry->vma->vm_mm, vaddr);
			if (!pgd_none(*pgd)) {
				pud = pud_offset(pgd, vaddr);
				if (!pud_none(*pud)) {
					pmd = pmd_offset(pud, vaddr);
					if (!pmd_none(*pmd)) {
						pte = pte_offset_map(pmd, vaddr);
						if (!pte_none(*pte)) {
							zap=1;
						}
					}
				}
			}

			/* Only zap a page if it falls within the mapped region
			 * and it has previously faulted
			 */
			if (zap && (addr_offset >= addr_start) &&
					(addr_offset < addr_end)) {

				if (!page_mapcount(pte_page(*pte))) {
					AGN_ERROR("ERROR No mapcount");
					AGN_DEBUG("ZI %p %08lX %d %d %p", pte_page(*pte),
					pte_page(*pte)->flags, page_count(pte_page(*pte)),
					page_mapcount(pte_page(*pte)), pte_page(*pte)->mapping);
				} else {
					atomic_add_negative(-1, &pte_page(*pte)->_mapcount);
					put_page(pte_page(*pte));
				}

				pte_clear(entry->vma->vm_mm, vaddr, pte);
			}

			if(pte) {
				pte_unmap(pte);
			}
		}
		up(&client_sem);
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	AGN_DEBUG("Exit");
	return 0;
}


int iegd_plb_remove_entries(struct agp_memory *mem,
	off_t pg_start, int type)
{
	int i, zap;
	struct list_head *tmp;
	struct client_list_struct *entry;
	unsigned long physaddr;
	unsigned long addr_start=0;
	unsigned long addr_end=0;
	unsigned long addr_offset=0;
	unsigned long vaddr;
	pgd_t *pgd;
	pud_t *pud;
	pmd_t *pmd;
	pte_t *pte;

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	for (i = pg_start; i < (mem->page_count + pg_start); i++) {
		if (i < private_data.gtt_entries) {
			physaddr = agp_bridge->gart_bus_addr + (i * PAGE_SIZE);
			physaddr |= 0x01;
			writel(physaddr, private_data.gtt+i);
			readl(private_data.gtt+i);	/* PCI Posting. */
		} else {
			if (private_data.split_gtt && (i >= 0x8000)) {
				writel(agp_bridge->scratch_page, private_data.upper_gtt + i - 0x8000);
		} else {
			writel(agp_bridge->scratch_page, private_data.gtt+i);
				readl(private_data.gtt+i);	/* PCI Posting. */
			}
		}

		down(&client_sem);
		list_for_each(tmp, &client_list) {
			entry = list_entry(tmp, struct client_list_struct, list);

			/* We need to handle invalidating VMA's that are only mapping
			 * a portion of the virtual aperture.  Calculate what if
			 * any invalidated pages need to be zapped
			 */
			addr_start = (entry->vma->vm_pgoff << PAGE_SHIFT)
				- agp_bridge->gart_bus_addr;
			addr_end = addr_start + (entry->vma->vm_end - entry->vma->vm_start);
			addr_offset = i << PAGE_SHIFT;

			vaddr = entry->vma->vm_start + (addr_offset - addr_start);

			zap=0;
			pgd=NULL;
			pud=NULL;
			pmd=NULL;
			pte=NULL;

			/* Look up page table entries for all VMAs that currently
			 * have the virtual aperture mapped -- to see if the page
			 * has ever faulted
			 */
			pgd = pgd_offset(entry->vma->vm_mm, vaddr);
			if (!pgd_none(*pgd)) {
				pud = pud_offset(pgd, vaddr);
				if (!pud_none(*pud)) {
					pmd = pmd_offset(pud, vaddr);
					if (!pmd_none(*pmd)) {
						pte = pte_offset_map(pmd, vaddr);
						if (!pte_none(*pte)) {
							zap=1;
						}
					}
				}
			}

			/* Only zap a page if it falls within the mapped region
			 * and it has previously faulted
			 */
			if (zap && (addr_offset >= addr_start) &&
					(addr_offset < addr_end)) {


				if (!page_mapcount(pte_page(*pte))) {
					AGN_ERROR("ERROR No mapcount");
					AGN_DEBUG("ZR %p %08lX %d %d %p", pte_page(*pte),
					pte_page(*pte)->flags, page_count(pte_page(*pte)),
					page_mapcount(pte_page(*pte)), pte_page(*pte)->mapping);
				} else {
					atomic_add_negative(-1, &pte_page(*pte)->_mapcount);
					put_page(pte_page(*pte));
				}

				pte_clear(entry->vma->vm_mm, vaddr, pte);
			}

			if(pte) {
				pte_unmap(pte);
			}
		}
		up(&client_sem);
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	return 0;
}


int iegd_plb_configure(void)
{
	struct aper_size_info_fixed *current_size;
	u32 temp;
	u16 gmch_ctrl;
	int i;

	current_size = A_SIZE_FIX(agp_bridge->current_size);

	/* SCH US15 uses the Base of Stolen Memory as it's artificial
	 * aperture address
	 */
	pci_read_config_dword(private_data.pdev, 0x5C, &temp);
	agp_bridge->gart_bus_addr = (temp & PCI_BASE_ADDRESS_MEM_MASK);

	pci_read_config_word(agp_bridge->dev,I830_GMCH_CTRL,&gmch_ctrl);
	gmch_ctrl |= I830_GMCH_ENABLED;
	pci_write_config_word(agp_bridge->dev,I830_GMCH_CTRL,gmch_ctrl);

	global_cache_flush();
	agp_bridge->driver->tlb_flush(0);

	writel(agp_bridge->gatt_bus_addr|I810_PGETBL_ENABLED,
	private_data.registers+I810_PGETBL_CTL);
	/* PCI Posting. */
	readl(private_data.registers+I810_PGETBL_CTL);

	if (agp_bridge->driver->needs_scratch_page) {

		for (i = private_data.gtt_entries; i < current_size->num_entries; i++) {

			writel(agp_bridge->scratch_page, private_data.gtt+i);
			readl(private_data.gtt+i);  /* PCI Posting. */
		}
	}

	global_cache_flush();

	return 0;
}


static void plb_destroy_pages(void *addr, size_t pg_count, unsigned int order)
{
	struct page *page;

	AGN_DEBUG("Enter");

	if (addr == NULL) {
		return;
	}

	page = virt_to_page(addr);
	change_page_attr(page, pg_count, PAGE_KERNEL);
	global_flush_tlb();
	put_page(page);
	AGP_UNLOCK_PAGE(page);

	if(page_count(page) > 1) {
		free_pages((unsigned long)addr, order);
	}

	atomic_dec(&agp_bridge->current_memory_agp);

	AGN_DEBUG("Exit");
}


void iegd_plb_free_by_type(struct agp_memory *curr)
{
	unsigned int order;

	switch (curr->page_count) {
	case 1:
		order = 0;   /* pg_count = 1 => 2 ^ 0 */
		break;
	case 4:
		order = 2;   /* pg_count = 4 => 2 ^ 2 */
		break;
	case 8:
		order = 3;   /* pg_count = 8 => 2 ^ 3 */
		break;
	default:
		/* This case should never happen */
		return;
	}

	agp_free_key(curr->key);
	if(curr->type == AGP_PHYS_MEMORY) {
		plb_destroy_pages(gart_to_virt(curr->memory[0]), curr->page_count, order);
		IGD_FREE_MEM(curr);
	}

	kfree(curr);

}

#endif
